/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/* This MX kernel lib code was originally contributed by
 * Brice.Goglin@ens-lyon.org (LIP/INRIA/ENS-Lyon) */

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx_klib_test.h"

char * send_pattern;
char * recv_pattern;

static int
alloc_patterns(void) {
  int i;
  send_pattern = alloc_buffer(128, MX_PIN_KERNEL);
  if (!send_pattern)
    return -1;
  for(i=0; i<128; i++)
    send_pattern[i] = i%128;

  recv_pattern = alloc_buffer(128, MX_PIN_KERNEL);
  if (!recv_pattern) {
    free_buffer(send_pattern, 128, MX_PIN_KERNEL);
    return -1;
  }
  for(i=0; i<128; i++)
    recv_pattern[i] = ((128-i)%128)+128;

  return 0;
}

static void
free_patterns(void) {
  if (send_pattern)
    free_buffer(send_pattern, 128, MX_PIN_KERNEL);
  if (recv_pattern)
    free_buffer(recv_pattern, 128, MX_PIN_KERNEL);
}


static int
one_test(mx_endpoint_t ep, mx_endpoint_addr_t addr, size_t len, int timeout,
	 int synch, int unexp, uintptr_t pin_type)
{
  char *send_buf = NULL;
  char *recv_buf = NULL;
  mx_request_t recv_req, send_req;
  mx_ksegment_t recv_seg[2], send_seg[2];
  mx_status_t status;
  uint32_t result;
  int ret;
  int send_done;

  if (len) {
    send_buf = alloc_buffer(len, pin_type);
    if (!send_buf)
      return -1;
    fill_pattern(send_buf, send_pattern, len, pin_type);

    recv_buf = alloc_buffer(len, pin_type);
    if (!recv_buf) {
      free_buffer(send_buf, len, pin_type);
      return -1;
    }
    fill_pattern(recv_buf, recv_pattern, len, pin_type);

    if (pin_type == MX_PIN_PHYSICAL) {
      send_seg[0].segment_ptr = MX_PA_TO_U64(virt_to_phys(send_buf));
      recv_seg[0].segment_ptr = MX_PA_TO_U64(virt_to_phys(recv_buf));
    } else if (pin_type == MX_PIN_KERNEL) {
      send_seg[0].segment_ptr = MX_KVA_TO_U64(send_buf);
      recv_seg[0].segment_ptr = MX_KVA_TO_U64(recv_buf);
    } else {
      send_seg[0].segment_ptr = MX_UVA_TO_U64(send_buf);
      recv_seg[0].segment_ptr = MX_UVA_TO_U64(recv_buf);
    }
  }
  send_seg[0].segment_length = len;
  recv_seg[0].segment_length = len;

  send_seg[1].segment_ptr = MX_KVA_TO_U64(NULL);
  recv_seg[1].segment_ptr = MX_KVA_TO_U64(NULL);
  send_seg[1].segment_length = 0;
  recv_seg[1].segment_length = 0;
  
  if (!unexp) {
    PRINT("posting the recv\n");
    ret = mx_kirecv(ep, recv_seg, 2, pin_type, 0, 0, NULL, &recv_req);
    if (ret) {
      PRINT("recv post failed %d\n", ret);
      return ret;
    }
    PRINT("recv successfully posted\n");
  }

  PRINT("posting the send\n");
  if (synch)
    ret = mx_kissend(ep, send_seg, 2, pin_type, addr, 0, NULL, &send_req);
  else
    ret = mx_kisend(ep, send_seg, 2, pin_type, addr, 0, NULL, &send_req);
  if (ret) {
    PRINT("send post failed %d\n", ret);
    return ret;
    }
  PRINT("send successfully posted\n");
  
  PRINT("waiting %d ms for send to complete\n", timeout);
  ret = mx_wait(ep, &send_req, timeout, &status, &result);
  if (ret) {
    PRINT("wait on send failed %d\n", ret);
    return ret;
    }
  send_done = result;
  if (result)
    PRINT("send completed with status %d\n", status.code);
  else
    PRINT("send still not completed\n");

  if (unexp) {
    PRINT("posting the recv\n");
    ret = mx_kirecv(ep, recv_seg, 2, pin_type, 0, 0, NULL, &recv_req);
    if (ret) {
      PRINT("recv post failed %d\n", ret);
      return ret;
    }
    PRINT("recv successfully posted\n");
  }  

  PRINT("waiting infinitely for irecv to complete\n");
  ret = mx_wait(ep, &recv_req, MX_INFINITE, &status, &result);
  if (ret) {
    PRINT("mx_wait on irecv failed %d\n", ret);
    return ret;
  }
  if (result)
    PRINT("recv completed with status %d\n", status.code);
  else
    PRINT("recv still not completed\n");

  if (!send_done) {
    PRINT("active polling until send completes\n");
    do { 
      ret = mx_test(ep, &send_req, &status, &result);
      if (ret) {
	PRINT("test on send failed %d\n", ret);
	return ret;
      }
    } while ( result != 1 );
    PRINT("send completed with status %d\n", status.code);
  }

  if (len) {
    if (compare(send_buf, recv_buf, len, pin_type)) {
      PRINT("Sent and received data are different !!!!\n");
      PRINT("0x%x vs 0x%x\n", *(int *)send_buf, *(int *)recv_buf);
      PRINT("0x%x vs 0x%x\n", *(int *)&send_buf[len - 4], *(int *)&recv_buf[len - 4]);
    } else {
      PRINT("No data corruption\n");
    }
  }

  if (len) {
    free_buffer(recv_buf, len, pin_type);
    free_buffer(send_buf, len, pin_type);
  }

  return 0;
}


static int
all_tests(int endpoint)
{
  mx_endpoint_t ep;
  mx_endpoint_addr_t addr;
  mx_return_t ret;
  int synch, unexp, lens[4] = {0, 16, 8192, 131072}, i;

  PRINT("\n");

  ret = mx_open_endpoint(MX_ANY_NIC, endpoint, 0, NULL, 0, &ep);
  if (ret) {
    PRINT("open failed %d\n", ret);
    return ret;
  }
  PRINT("open successful\n");

  ret = mx_get_endpoint_addr(ep, &addr);
  if (ret) {
    PRINT("mx_get_endpoint_addr failed %d\n", ret);
    return ret;
  }
  PRINT("mx_get_endpoint_addr successful\n");

  PRINT("Starting TEST\n\n");

  if (alloc_patterns() < 0) {
    PRINT("alloc patterns failed\n");
    goto error;
  }

  for(i=0; i<4; i++) {
    int len = lens[i];
    for(synch=0; synch<2; synch++)
      for(unexp=0; unexp<2; unexp++) {
	PRINT("testing length %d %ssynch %sexpected MX_PIN_KERNEL...\n", len,
	      synch ? "" : "a",
	      unexp ? "un" : "");
	if (one_test(ep, addr, len, 2000, synch, unexp, MX_PIN_KERNEL) < 0)
	  goto error;
	PRINT("\n");
      }
  }

  for(i=0; i<4; i++) {
    int len = lens[i];
    for(synch=0; synch<2; synch++)
      for(unexp=0; unexp<2; unexp++) {
	PRINT("testing length %d %ssynch %sexpected MX_PIN_PHYSICAL...\n", len,
	      synch ? "" : "a",
	      unexp ? "un" : "");
	if (one_test(ep, addr, len, 2000, synch, unexp, MX_PIN_PHYSICAL) < 0)
	  goto error;
	PRINT("\n");
      }
  }

  for(i=0; i<4; i++) {
    int len = lens[i];
    for(synch=0; synch<2; synch++)
      for(unexp=0; unexp<2; unexp++) {
	PRINT("testing length %d %ssynch %sexpected MX_PIN_USER...\n", len,
	      synch ? "" : "a",
	      unexp ? "un" : "");
	if (one_test(ep, addr, len, 2000, synch, unexp, MX_PIN_USER) < 0)
	  goto error;
	PRINT("\n");
      }
  }

  free_patterns();

  ret = mx_close_endpoint(ep);
  if (ret) {
    PRINT("close failed %d\n", ret);
    return ret;
  }
  PRINT("close successful\n");

  return 0;

 error:
  free_patterns();
  ret = mx_close_endpoint(ep);
  return -1;
}

static int
klib_test_init_module(void)
{
  int ret;

  mx_init();
  ret = all_tests(2);
  if (ret)
    return ret;
  return 0;
}

static void
klib_test_cleanup_module(void)
{
  mx_finalize();
}

DO_MODULE(klib_test_init_module, klib_test_cleanup_module);
